////////////////////////////////////////////////////////////////////////////////
//
//                        Borland Delphi Runtime Library
//                      Planet of Death (POD) File Classes
//
// The original file is: PodFiles.pas, released 2004-09-01.
// The initial developer of the code is: Nico Bendlin <nicode@gmx.net>.
//
// Portions created by Nico Bendlin are
// Copyright (c) 2004 Nico Bendlin.
//
// The contents of this file are used with permission, subject to the Mozilla
// Public License Version 1.1 (the "License"); you may not use this file except
// in compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/MPL-1.1.html
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
// the specific language governing rights and limitations under the License.
//
// Alternatively, the contents of this file may be used under the terms of
// either the GNU General Public License Version 2 or later (the "GPL"), or
// the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
// in which case the provisions of the GPL or the LGPL are applicable instead
// of those above. If you wish to allow use of your version of this file only
// under the terms of either the GPL or the LGPL, and not to allow others to
// use your version of this file under the terms of the MPL, indicate your
// decision by deleting the provisions above and replace them with the notice
// and other provisions required by the GPL or the LGPL. If you do not delete
// the provisions above, a recipient may use your version of this file under
// the terms of any one of the MPL, the GPL or the LGPL.
//
////////////////////////////////////////////////////////////////////////////////
//
//  Notes:  First public version.  Not yet optimized for speed - but works :)
//
////////////////////////////////////////////////////////////////////////////////

unit PodFiles;

{.$HINTS OFF}
{.$WARNINGS OFF}
{$RANGECHECKS OFF}
{$OVERFLOWCHECKS OFF}
{$EXTENDEDSYNTAX ON}

interface

uses
  Classes, PodConst;


////////////////////////////////////////////////////////////////////////////////
//
//  POD Binary Data File (PBDF) Stream
//
//    Generic memory stream for loading and writing PBDFs.  It includes
//    autodetection for blocklength and ecryption keys.  Decryption and
//    encryption of demo version tracks (BL8 and BL9) are included too.
//
//    Tested PBDF formats/versions:  BL3, BL4, BL6, BL8, BL9, BV3, BV4,
//     BV6, BV8, BV9, CH3, CH4, BP3, BI2, BMD, BBM, BVB, BRE, BRH, BRT.
//

type
  TPodBdfStream = class(TStream)
  private
    FMemory: Pointer;
    FSize: Longint;
    FPosition: Longint;
    FCapacity: Longint;
    FXorKey: Longword;
    FBlockLength: Longint;
    procedure SetCapacity(NewCapacity: Longint);
  protected
    procedure SetPointer(Ptr: Pointer; Size: Longint);
    function Realloc(var NewCapacity: Longint): Pointer; virtual;
    property Capacity: Longint read FCapacity write SetCapacity;
  public
    destructor Destroy; override;
    function Read(var Buffer; Count: Longint): Longint; override;
    function Seek(Offset: Longint; Origin: Word): Longint; override;
    procedure SaveToStream(Stream: TStream);
    procedure SaveToFile(const FileName: string);
    property Memory: Pointer read FMemory;
    procedure Clear;
    procedure LoadFromStream(Stream: TStream);
    procedure LoadFromFile(const FileName: string);
    procedure SetSize(NewSize: Longint); override;
    function Write(const Buffer; Count: Longint): Longint; override;
    property XorKey: Longword read FXorKey write FXorKey;
    property BlockLength: Longint read FBlockLength write FBlockLength;
    procedure Reset;
  end;

  
////////////////////////////////////////////////////////////////////////////////
//
//  Well-known PBDF file types
//

type
  TPodFileType = (
    pftUnknown,
    pftCircuit,
    pftVoiture,
    pftChrome,
    pftPolice,
    pftShadow,
    pftMessage,
    pftBitmap,
    pftVideo,
    pftRecord
  );

const
  PodFileTypesNames: array [TPodFileType] of string = (
    SPodUnknown,
    SPodCircuit,
    SPodVoiture,
    SPodChrome,
    SPodPolice,
    SPodShadow,
    SPodMessage,
    SPodBitmap,
    SPodVideo,
    SPodRecord
  );


////////////////////////////////////////////////////////////////////////////////
//
//  Well-known PBDF XorKeys and BlockLengths
//

const
  // pftCircuit
  PodBl3Key = $00000F7E;
  PodBl4Key = $00000F7E;
  PodBl6Key = $00000F7E;
  PodBl8Key = $0000D13F;  // blocks > 0 are specially encrypted (POD Demo)
  PodBl9Key = $0000D13F;  // blocks > 0 are specially encrypted (POD Demo)
  PodBl3Len = $00001000;
  PodBl4Len = $00001000;
  PodBl6Len = $00001000;
  PodBl8Len = $00000900;
  PodBl9Len = $00000900;
  // pftVoiture
  PodBv3Key = $00000F2E;
  PodBv4Key = $00000F2E;
  PodBv6Key = $00000F2E;
  PodBv8Key = $0000EFA9;
  PodBv9Key = $0000EFA9;
  PodBv3Len = $00001000;
  PodBv4Len = $00001000;
  PodBv6Len = $00001000;
  PodBv8Len = $00000600;
  PodBv9Len = $00000600;
  // pftChrome
  PodCh3Key = $00000DD6;
  PodCh4Key = $00000DD6;
  PodCh3Len = $00001000;
  PodCh4Len = $00001000;
  // pftPolice
  PodBp3Key = $00000FAE;
  PodBp3Len = $00001000;
  // pftShadow
  PodBi2Key = $00000F7E;
  PodBi2Len = $00001000;
  // pftMessage
  PodBmdKey = $0000EA1E;
  PodBmdLen = $00000080;
  // pftBitmap
  PodBbmKey = $00000F3A;
  PodBbmLen = $00000800;
  // pftVideo
  PodBvbKey = $00000F3A;
  PodBvbLen = $00000800;
  // pftRecord
  PodBreKey = $0000EAFA;
  PodBrhKey = $0000EAFA;
  PodBrtKey = $0000EAFA;
  PodBreLen = $00000400;
  PodBrhLen = $00000400;
  PodBrtLen = $00000400;

type
  TPodFileVers = Byte;
  TPodFileInfo = record
    Typ: TPodFileType;
    Ver: TPodFileVers;
    Ext: string;
    Key: Longword;
    Len: Longword;
  end;

const
  PodFileList: array [0..20] of TPodFileInfo = (
    (Typ: pftUnknown; Ver: 0; Ext: '.*';   Key: $00000000; Len: $00000000),
    (Typ: pftCircuit; Ver: 3; Ext: '.bl3'; Key: PodBl3Key; Len: PodBl3Len),
    (Typ: pftCircuit; Ver: 4; Ext: '.bl4'; Key: PodBl4Key; Len: PodBl4Len),
    (Typ: pftCircuit; Ver: 6; Ext: '.bl6'; Key: PodBl6Key; Len: PodBl6Len),
    (Typ: pftCircuit; Ver: 8; Ext: '.bl8'; Key: PodBl8Key; Len: PodBl8Len),
    (Typ: pftCircuit; Ver: 9; Ext: '.bl9'; Key: PodBl9Key; Len: PodBl9Len),
    (Typ: pftVoiture; Ver: 3; Ext: '.bv3'; Key: PodBv3Key; Len: PodBv3Len),
    (Typ: pftVoiture; Ver: 4; Ext: '.bv4'; Key: PodBv4Key; Len: PodBv4Len),
    (Typ: pftVoiture; Ver: 6; Ext: '.bv6'; Key: PodBv6Key; Len: PodBv6Len),
    (Typ: pftVoiture; Ver: 8; Ext: '.bv8'; Key: PodBv8Key; Len: PodBv8Len),
    (Typ: pftVoiture; Ver: 9; Ext: '.bv9'; Key: PodBv9Key; Len: PodBv9Len),
    (Typ: pftChrome;  Ver: 3; Ext: '.ch3'; Key: PodCh3Key; Len: PodCh3Len),
    (Typ: pftChrome;  Ver: 4; Ext: '.ch4'; Key: PodCh4Key; Len: PodCh4Len),
    (Typ: pftPolice;  Ver: 3; Ext: '.bp3'; Key: PodBp3Key; Len: PodBp3Len),
    (Typ: pftShadow;  Ver: 2; Ext: '.bi2'; Key: PodBi2Key; Len: PodBi2Key),
    (Typ: pftMessage; Ver: 0; Ext: '.bmd'; Key: PodBmdKey; Len: PodBmdLen),
    (Typ: pftBitmap;  Ver: 0; Ext: '.bbm'; Key: PodBbmKey; Len: PodBbmLen),
    (Typ: pftVideo;   Ver: 0; Ext: '.bvb'; Key: PodBvbKey; Len: PodBvbLen),
    (Typ: pftRecord;  Ver: 0; Ext: '.bre'; Key: PodBreKey; Len: PodBreLen),
    (Typ: pftRecord;  Ver: 0; Ext: '.brh'; Key: PodBrhKey; Len: PodBrhLen),
    (Typ: pftRecord;  Ver: 0; Ext: '.brt'; Key: PodBrtKey; Len: PodBrtLen)
  );

function GetPodFileInfo(const FileName: string): TPodFileInfo;
function GetPodFileType(const FileName: string): TPodFileType;
function GetPodFileVers(const FileName: string): TPodFileVers;


implementation

uses
  SysUtils;


////////////////////////////////////////////////////////////////////////////////
//
//  TPodBdfStream
//

procedure TPodBdfStream.SetCapacity(NewCapacity: Longint);
begin
  SetPointer(Realloc(NewCapacity), FSize);
  FCapacity := NewCapacity;
end;

procedure TPodBdfStream.SetPointer(Ptr: Pointer; Size: Longint);
begin
  FMemory := Ptr;
  FSize := Size;
end;

function TPodBdfStream.Realloc(var NewCapacity: Longint): Pointer;
const
  MemoryDelta = $1000;
begin
  if (NewCapacity > 0) then
    NewCapacity := (NewCapacity + (MemoryDelta - 1)) and not (MemoryDelta - 1);
  Result := Memory;
  if (NewCapacity <> FCapacity) then
  begin
    if (NewCapacity = 0) then
    begin
      FreeMem(Memory);
      Result := nil;
    end
    else
    begin
      if (Capacity = 0) then
        Result := AllocMem(NewCapacity)
      else
        Result := ReallocMemory(Memory, NewCapacity);
      if (Result = nil) then
        raise EStreamError.CreateRes(PResStringRec(@SPodMemoryStreamError));
    end;
  end;
end;

destructor TPodBdfStream.Destroy;
begin
  Clear;
  inherited Destroy;
end;

function TPodBdfStream.Read(var Buffer; Count: Longint): Longint;
begin
  if (FPosition >= 0) and (Count >= 0) then
  begin
    Result := FSize - FPosition;
    if (Result > 0) then
    begin
      if (Result > Count) then
        Result := Count;
      Move(Pointer(Integer(FMemory) + FPosition)^, Buffer, Result);
      Inc(FPosition, Result);
      Exit;
    end;
  end;
  Result := 0;
end;

function TPodBdfStream.Seek(Offset: Longint; Origin: Word): Longint;
begin
  case Origin of
    soFromBeginning:
      FPosition := Offset;
    soFromCurrent:
      Inc(FPosition, Offset);
    soFromEnd:
      FPosition := FSize + Offset;
  end;
  Result := FPosition;
end;

procedure TPodBdfStream.SaveToStream(Stream: TStream);
begin
  if (FSize <> 0) then
    Stream.WriteBuffer(FMemory^, FSize);
end;

procedure TPodBdfStream.SaveToFile(const FileName: string);
var
  Stream: TStream;
  MemPtr: PLongWord;
  Chksum: Longword;
  Block: Integer;
  Index: Integer;
  Value: Longword;
  Bl9Key: Longword;
  Bl9Val: Longword;
begin
  Stream := TFileStream.Create(FileName, fmCreate);
  try
    // No blocks (no checksums)
    if (FBlockLength <= 0) then
    begin
      // No encryption
      if (FXorKey = 0) then
        SaveToStream(Stream)
      else
      begin
        // Encrytion only
        MemPtr := FMemory;
        for Index := 0 to FSize div SizeOf(Longword) - 1 do
        begin
          Value := MemPtr^ xor FXorKey;
          Stream.WriteBuffer(Value, SizeOf(Longword));
          Inc(MemPtr);
        end;
      end;
    end
    else
    begin
      // Encrypted with checksums
      MemPtr := FMemory;
      for Block := 0 to
        FSize div ((FBlockLength - 1) * SizeOf(Longword)) - 1 do
      begin
        // Special handling for BL8/BL9
        if (FXorKey = PodBl9Key) and (Block > 0) then
        begin
          Bl9Val := 0;
          Chksum := 0;
          for Index := 1 to FBlockLength - 1 do
          begin
            Value := MemPtr^;
            Inc(Chksum, Value);
            case (Bl9Val shr 16 and 3) of
              $1: Bl9Key := ($3AF70BC4 - Bl9Val);
              $2: Bl9Key := (Bl9Val + $07091971) shl 1;
              $3: Bl9Key := ($11E67319 - Bl9Val) shl 1;
            else
              Bl9Key := (Bl9Val - $50A4A89D);
            end;
            case (Bl9Val and 3) of
              1: Bl9Val := not Value xor not Bl9Key;
              2: Bl9Val := Value xor not Bl9Key;
              3: Bl9Val := Value xor Bl9Key xor $FFFF;
            else
              Bl9Val := not Value xor Bl9Key;
            end;
            Stream.WriteBuffer(Bl9Val, SizeOf(Bl9Val));
            Inc(MemPtr);
          end;
          Stream.WriteBuffer(Chksum, SizeOf(Chksum));
        end
        else
        begin
          // Default is XOR with Checksum
          Chksum := 0;
          for Index := 1 to FBlockLength - 1 do
          begin
            Inc(Chksum, MemPtr^);
            Value := MemPtr^ xor FXorKey;
            Stream.WriteBuffer(Value, SizeOf(Longword));
            Inc(MemPtr);
          end;
          Stream.WriteBuffer(Chksum, SizeOf(Longword));
        end;
      end;
    end;
  finally
    Stream.Free;
  end;
end;

procedure TPodBdfStream.Clear;
begin
  SetCapacity(0);
  FSize := 0;
  FPosition := 0;
end;

procedure TPodBdfStream.LoadFromStream(Stream: TStream);
var
  Count: Longint;
begin
  Stream.Position := 0;
  Count := Stream.Size;
  SetSize(Count);
  if (Count <> 0) then
    Stream.ReadBuffer(FMemory^, Count);
end;

procedure TPodBdfStream.LoadFromFile(const FileName: string);
var
  Stream: TStream;
  Blocks: Integer;
  Chksum: Longword;
  Value: Longword;
  Index: Integer;
  Block: Longword;
  Bl9Key: Longword;
  Bl9Old: Longword;
  Bl9New: Longword;
begin
  Stream := TFileStream.Create(FileName, fmOpenRead or fmShareDenyWrite);
  try
    Clear;
    if (Stream.Read(Value, SizeOf(Value)) = SizeOf(Value)) then
    begin
      // Auto detect if XorKey if 0
      if (FXorKey = 0) then
        FXorKey := Value xor Longword(Stream.Size);
      Blocks := 0;
      // Read first block (-1: all)
      if (FBlockLength <= 0) then
      begin
        Chksum := 0;
        for Index := 1 to Stream.Size div SizeOf(Value) do
        begin
          // Auto detect BlockLength if 0
          if (FBlockLength = 0) and (Index mod $80 = 0) and
            (Stream.Size mod Index = 0) and (Value = Chksum) then
          begin
            FBlockLength := Index;
            Inc(Blocks);
            Break;
          end;
          Value := Value xor FXorKey;
          Inc(Chksum, Value);
          WriteBuffer(Value, SizeOf(Longword));
          if (Index < Stream.Size div SizeOf(Value)) then
            Stream.ReadBuffer(Value, SizeOf(Longword));
        end;
        // Not auto detected or not found
        if (FBlockLength <= 0) then
        begin
          Value := Value xor FXorKey;
          WriteBuffer(Value, SizeOf(Longword));
        end;
      end;
      // Read the remaining blocks if any
      if (FBlockLength > 0) then
      begin
        for Block := Blocks to
          Stream.Size div (FBlockLength * SizeOf(Value)) - 1 do
        begin
          // Special handling for BL8/BL9
          if (FXorKey = PodBl9Key) and (Block > 0) then
          begin
            Chksum := 0;
            Bl9Old := 0;
            for Index := 1 to FBlockLength - 1 do
            begin
              Stream.ReadBuffer(Value, SizeOf(Value));
              case (Bl9Old shr 16 and 3) of
                1: Bl9Key := ($3AF70BC4 - Bl9Old);
                2: Bl9Key := (Bl9Old + $07091971) shl 1;
                3: Bl9Key := ($11E67319 - Bl9Old) shl 1;
              else
                Bl9Key := (Bl9Old - $50A4A89D);
              end;
              case (Bl9Old and 3) of
                0: Bl9New := not Value xor Bl9Key;
                1: Bl9New := not Value xor not Bl9Key;
                2: Bl9New := Value xor not Bl9Key;
                3: Bl9New := Value xor bl9Key xor $FFFF;
              end;
              Bl9Old := Value;
              WriteBuffer(Bl9New, SizeOf(Bl9New));
              Inc(Chksum, Bl9New);
            end;
            Stream.ReadBuffer(Value, SizeOf(Value));
            if (Value <> Chksum) then
              raise EStreamError.CreateResFmt(
                PResStringRec(@SPodChksumStreamError), [FileName]);
          end
          else
          begin
            // Default is XOR with Checksum
            Chksum := 0;
            for Index := 0 to FBlockLength - 2 do
            begin
              Stream.ReadBuffer(Value, SizeOf(Value));
              Value := Value xor FXorKey;
              Inc(Chksum, Value);
              WriteBuffer(Value, SizeOf(Value));
            end;
            Stream.ReadBuffer(Value, SizeOf(Value));
            if (Value <> Chksum) then
              raise EStreamError.CreateResFmt(
                PResStringRec(@SPodChksumStreamError), [FileName]);
          end;
        end;
      end;
    end;
  finally
    Stream.Free;
  end;
end;

procedure TPodBdfStream.SetSize(NewSize: Longint);
var
  OldPosition: Longint;
begin
  OldPosition := FPosition;
  SetCapacity(NewSize);
  FSize := NewSize;
  if (OldPosition > NewSize) then
    Seek(0, soFromEnd);
end;

function TPodBdfStream.Write(const Buffer; Count: Longint): Longint;
var
  Pos: Longint;
begin
  if (FPosition >= 0) and (Count >= 0) then
  begin
    Pos := FPosition + Count;
    if (Pos > 0) then
    begin
      if (Pos > FSize) then
      begin
        if (Pos > FCapacity) then
          SetCapacity(Pos);
        FSize := Pos;
      end;
      System.Move(Buffer, Pointer(Integer(FMemory) + FPosition)^, Count);
      FPosition := Pos;
      Result := Count;
      Exit;
    end;
  end;
  Result := 0;
end;

procedure TPodBdfStream.Reset;
begin
  Clear;
  FXorKey := 0;
  BlockLength := 0;
end;

////////////////////////////////////////////////////////////////////////////////
//
//  GetPodFileInfo
//

function GetPodFileInfo(const FileName: string): TPodFileInfo;
var
  FileExt: string;
  Index: Integer;
begin
  Result := PodFileList[0];
  FileExt := ExtractFileExt(FileName);
  for Index := Low(PodFileList) to High(PodFileList) do
    if (CompareText(PodFileList[Index].Ext, FileExt) = 0) then
    begin
      Result := PodFileList[Index];
      Break;
    end;
end;

function GetPodFileType(const FileName: string): TPodFileType;
var
  FileExt: string;
  Index: Integer;
begin
  Result := pftUnknown;
  FileExt := ExtractFileExt(FileName);
  for Index := Low(PodFileList) to High(PodFileList) do
    if (CompareText(PodFileList[Index].Ext, FileExt) = 0) then
    begin
      Result := PodFileList[Index].Typ;
      Break;
    end;
end;

function GetPodFileVers(const FileName: string): TPodFileVers;
var
  FileExt: string;
  Index: Integer;
begin
  Result := 0;
  FileExt := ExtractFileExt(FileName);
  for Index := Low(PodFileList) to High(PodFileList) do
    if (CompareText(PodFileList[Index].Ext, FileExt) = 0) then
    begin
      Result := PodFileList[Index].Ver;
      Break;
    end;
end;

end.
