unit PodPaintMain;

interface

uses
  Windows, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ComCtrls, ExtCtrls,
  PodFiles, PodUtils, ExtDlgs;

type
  PPodTexDesc = ^TPodTexDesc;
  TPodTexDesc = packed record
    Name  : array [0..31] of AnsiChar;
    Left  : Longword;
    Top   : Longword;
    Right : Longword;
    Bottom: Longword;
    Index : Longword;
  end;

type
  TPodTexture = packed record
    Width  : Longint;
    Height : Longint;
    PalData: Longint;
    PixData: Longint;
    TexDesc: array of TPodTexDesc;
  end;

type
  TMainForm = class(TForm)
    MainPanel: TPanel;
    FilePanel: TPanel;
    OpenEdit: TEdit;
    OpenButton: TButton;
    OpenDialog: TOpenDialog;
    DumpButton: TButton;
    DumpDialog: TSaveDialog;
    SaveButton: TButton;
    SaveDialog: TSaveDialog;
    TexsLabel: TLabel;
    TexsTreeView: TTreeView;
    ExitButton: TButton;
    FileSplitter: TSplitter;
    ViewPanel: TPanel;
    ViewImage: TImage;
    RectImage: TImage;
    ImpoButton: TButton;
    ImpoDialog: TOpenPictureDialog;
    ExpoButton: TButton;
    ExpoDialog: TSavePictureDialog;
    MainSplitter: TSplitter;
    InfoMemo: TMemo;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ExitButtonClick(Sender: TObject);
    procedure OpenButtonClick(Sender: TObject);
    procedure DumpButtonClick(Sender: TObject);
    procedure SaveButtonClick(Sender: TObject);
    procedure TexsTreeViewChange(Sender: TObject; Node: TTreeNode);
    procedure ImpoButtonClick(Sender: TObject);
    procedure ImpoDialogTypeChange(Sender: TObject);
    procedure ExpoButtonClick(Sender: TObject);
    procedure ExpoDialogTypeChange(Sender: TObject);
  private
    BdfStream: TPodBdfStream;
    Textures: array of TPodTexture;
  end;

var
  MainForm: TMainForm;

implementation

{$R *.dfm}

procedure TMainForm.FormCreate(Sender: TObject);
var
  Filter: string;
  Index: Integer;
  Mask: string;
  Loop: TPodFileType;
begin
  BdfStream := TPodBdfStream.Create;
  OpenEdit.Text := EmptyStr;
  Filter := 'POD Binary Data Files (';
  Mask := '';
  for Index := Low(PodFileList) to High(PodFileList) do
    if (PodFileList[Index].Typ <> pftUnknown) then
    begin
      if (Length(Mask) > 0) then
        Mask := Mask + ';';
      Mask := Mask + '*' + PodFileList[Index].Ext;
    end;
  Filter := Filter + Mask + ')|' + Mask;
  for Loop := Low(TPodFileType) to High(TPodFileType) do
    if (Loop <> pftUnknown) then
    begin
      Filter := Filter + '|POD ' + PodFileTypesNames[Loop] + ' (';
      Mask := '';
      for Index := Low(PodFileList) to High(PodFileList) do
        if (PodFileList[Index].Typ = Loop) then
        begin
          if (Length(Mask) > 0) then
            Mask := Mask + ';';
          Mask := Mask + '*' + PodFileList[Index].Ext;
        end;
      Filter := Filter + Mask + ')|' + Mask;
    end;
  Filter := Filter + '|All Files (*.*)|*';
  OpenDialog.Filter := Filter;
  SaveDialog.Filter := Filter;
  TexsTreeView.Items.Clear();
  DumpButton.Enabled := False;
  SaveButton.Enabled := False;
  InfoMemo.Clear();
  ImpoButton.Enabled := False;
  ExpoButton.Enabled := False;
end;

procedure TMainForm.FormDestroy(Sender: TObject);
begin
  BdfStream.Free;
end;

procedure TMainForm.ExitButtonClick(Sender: TObject);
begin
  Close;
end;

procedure TMainForm.OpenButtonClick(Sender: TObject);
var
  Texture: Longint;
  Palette: Longint;
  PodInfo: TPodFileInfo;
  Search: PLongWord;
  Data: Pointer;
  Desc: PPodTexDesc;
  Block: Integer;
  Blocks: Integer;
  Mats: LongWord;
  Loop: Integer;
  Flags: LongWord;
  Found: Integer;
  Skies: Integer;
  NodeName: string;
  TreeNode: TTreeNode;
  FSize: Extended;
  function MemoryToOffset(Data: Pointer): Longint;
  begin
    Result := Longint(Integer(Data) - Integer(BdfStream.Memory));
  end;
begin
  if (not OpenDialog.Execute()) then
    Exit;
  OpenEdit.Text := ExtractFileName(OpenDialog.FileName);
  OpenEdit.Hint := OpenDialog.FileName;
  OpenButton.Enabled := False;
  DumpButton.Enabled := False;
  DumpDialog.FileName := ExtractFileName(OpenDialog.FileName) + '.dmp';
  DumpDialog.InitialDir := ExtractFilePath(OpenDialog.FileName);
  SaveButton.Enabled := False;
  SaveDialog.FilterIndex := OpenDialog.FilterIndex;
  SaveDialog.FileName := ExtractFileName(OpenDialog.FileName);
  SaveDialog.InitialDir := ExtractFilePath(OpenDialog.FileName);
  TexsTreeView.Items.BeginUpdate();
  TexsTreeView.Items.Clear();
  InfoMemo.Lines.BeginUpdate();
  InfoMemo.Clear();
  ImpoButton.Enabled := False;
  ImpoDialog.InitialDir := ExtractFilePath(OpenDialog.FileName);
  ExpoButton.Enabled := False;
  ExpoDialog.InitialDir := ExtractFilePath(OpenDialog.FileName);
  ViewImage.Picture.Bitmap.Handle := 0;
  RectImage.Picture.Bitmap.Handle := 0;
  BdfStream.Reset();
  for Texture := Low(Textures) to High(Textures) do
    SetLength(Textures[Texture].TexDesc, 0);
  SetLength(Textures, 0);
  Texture := -1;
  try
    BdfStream.LoadFromFile(OpenDialog.FileName);
    try
      with InfoMemo.Lines do
      begin
        Add(EmptyStr);
        with BdfStream do
        begin
          Add('  XorKey     : $' + IntToHex(XorKey, 8));
          Add('  BlockLength: $' + IntToHex(BlockLength, 8));
          FSize := Size;
          Add('  Size       : $' + IntToHex(Size, 8) +
            ' (' + Format('%.0n', [FSize]) + ' Bytes)');
        end;
      end;
      PodInfo := GetPodFileInfo(OpenDialog.FileName);
      with InfoMemo.Lines do
      begin
        Add(EmptyStr);
        with PodInfo do
        begin
          Add('  Typ: ' + PodFileTypesNames[Typ]);
          Add('  Ver: ' + IntToStr(Ver));
          Add('  Ext: ''' + Ext + '''');
          Add('  Key: $' + IntToHex(Key, 8));
          Add('  Len: $' + IntToHex(Len, 8));
        end;
      end;
      Search := BdfStream.Memory;
      Found := 0;
      Skies := 0;
      while (Cardinal(Search) < (Cardinal(BdfStream.Memory) +
        BdfStream.Size - SizeOf(TPodTexDesc) - 1)) do
      begin
        if (Search^ or $20202000 = $6167742E) or // '.tga', '.TGA', ...
          (Search^ or $20202000 = $706D622E) or  // '.bmp', '.BMP', ...
          (Search^ = $5F796B73) then             // 'sky_' (Test.BLx)
        begin
          Data := Search;
          while (PAnsiChar(Data)[0] <> #0) do
            Dec(Cardinal(Data));
          Dec(Cardinal(Data), $0B);
          Blocks := PLongWord(Data)^;
          Inc(Cardinal(Data), $04);
          Flags := PLongWord(Data)^;
          Inc(Cardinal(Data), $04);
          if Blocks > 0 then
          begin
            Inc(Texture, Blocks);
            SetLength(Textures, Texture + 1);
            Block := 1;
            while (Block <= Blocks) do
            begin
              Mats := PLongWord(Data)^;
              with Textures[Texture - Blocks + Block] do
              begin
                SetLength(TexDesc, Mats);
                Inc(Cardinal(Data), SizeOf(Longword));
                Desc := Data;
                Loop := 0;
                while (Loop < Integer(Mats)) do
                begin
                  TexDesc[Loop] := Desc^;
                  TexDesc[Loop].Name[13] := #0;
                  Inc(Desc);
                  Inc(Loop);
                end;
              end;
              Data := Desc;
              Inc(Block);
            end;
            Search := Data;
            if (PodInfo.Ver in [3, 6, 8]) then
            begin
              Palette := MemoryToOffset(Data);
              Inc(Cardinal(Data), SizeOf(TRGBTriple) * 256);
            end
            else
              Palette := 0;
            Block := 1;
            while (Block <= Blocks) do
            begin
              with Textures[Texture - Blocks + Block] do
              begin
                if (PodInfo.Ver in [3, 6, 8]) then
                begin
                  Width := $100;
                  Height := $100;
                  PalData := Palette;
                  PixData := MemoryToOffset(Data);
                  Inc(Cardinal(Data), Width * Height);
                end
                else
                begin
                  // Hackorama...
                  if (PodInfo.Typ in [pftVoiture, pftChrome])  then
                  begin
                    Width := $80;
                    Height := $80;
                  end
                  else
                  begin
                    Width := $100;
                    Height := $100;
                    if Flags > 0 then
                    begin
                      if (Skies > 0) then
                      begin
                        Width := $80;
                        Height := $80;
                      end;
                      Inc(Skies);
                    end
                    else if (Found > 0) then
                    begin
                      Width := $80;
                      Height := $80;
                    end;
                  end;
                  // Slimepit Hack
                  if (High(TexDesc) = 0) and
                    (TexDesc[0].Right = $003F) and
                    (TexDesc[0].Bottom = $03FF) then
                  begin
                    Width := $100;
                    Height := $100;
                    TexDesc[0].Right := $FF;
                    TexDesc[0].Bottom := $FF;
                  end;
                  PalData := 0;
                  PixData := MemoryToOffset(Data);
                  Inc(Cardinal(Data), Width * Height * SizeOf(Word));
                end;
              end;
              Inc(Block);
            end;
          end;
          Inc(Found);
        end;
        Inc(Cardinal(Search));
      end;
      for Loop := Low(Textures) to High(Textures) do
      begin
        NodeName := ExtractFileName(OpenDialog.FileName);
(** )
        SetLength(NodeName, Length(NodeName) - Length(ExtractFileExt(NodeName)));
        NodeName := NodeName + '_' + IntToStr(Loop);
(**)
        NodeName := NodeName + '.' + IntToStr(Loop);
(**)
        TreeNode := TexsTreeView.Items.AddObject(nil, NodeName, Pointer(Loop));
        with Textures[Loop] do
          for Block := Low(TexDesc) to High(TexDesc) do
          begin
            TexsTreeView.Items.AddChildObject(TreeNode,
              TexDesc[Block].Name, Pointer(Loop));
          end;
      end;
    finally
      SaveButton.Enabled := (BdfStream.BlockLength > 0) and
        (BdfStream.XorKey and $FFFF0000 = 0);
      DumpButton.Enabled := True;
    end;
  finally
    TexsTreeView.Items.EndUpdate();
    InfoMemo.Lines.EndUpdate();
    OpenButton.Enabled := True;
  end;
end;

procedure TMainForm.DumpButtonClick(Sender: TObject);
var
  MemStream: TMemoryStream;
begin
  if (not DirectoryExists(OpenDialog.InitialDir)) then
    OpenDialog.InitialDir := EmptyStr;
  if (not DumpDialog.Execute()) then
    Exit;
  MemStream := TMemoryStream.Create();
  try
    BdfStream.SaveToStream(MemStream);
    MemStream.SaveToFile(DumpDialog.FileName);
  finally
    MemStream.Free;
  end;
end;

procedure TMainForm.SaveButtonClick(Sender: TObject);
begin
  with SaveDialog do
  begin
    if (not DirectoryExists(InitialDir)) then
      InitialDir := EmptyStr;
    if (not Execute()) then
      Exit;
    if FileExists(FileName) and not FileExists(FileName + '.bak') then
      MoveFile(PChar(FileName), PChar(FileName + '.bak'));
  end;
  BdfStream.SaveToFile(SaveDialog.FileName);
end;

procedure TMainForm.TexsTreeViewChange(Sender: TObject; Node: TTreeNode);
var
  Idx: Integer;
  Pal: TMemoryStream;
  Pix: TMemoryStream;
  Rgb: PRGBTriple;
  Dat: PByte;
  Dsc: TPodTexDesc;
begin
  InfoMemo.Clear();
  ViewImage.Picture.Bitmap.Handle := 0;
  RectImage.Picture.Bitmap.Handle := 0;
  if (Node = nil) then
    Exit;
  Idx := Integer(Node.Data);
  if (Idx < Low(Textures)) or (Idx > High(Textures)) then
    Exit;
  Pal := TMemoryStream.Create();
  Pix := TMemoryStream.Create();
  InfoMemo.Lines.BeginUpdate();
  ImpoButton.Enabled := False;
  ExpoButton.Enabled := False;
  try
    with Textures[Idx] do
    begin
      with InfoMemo.Lines do
      begin
        Clear();
        Add(EmptyStr);
        with Textures[Idx] do
        begin
          if (Node.Parent <> nil) then
          begin
            with TexDesc[Node.Index] do
            begin
              Add('  Name  : ''' + Name + '''');
              Add('  Left  : $' + IntToHex(Left, 8));
              Add('  Top   : $' + IntToHex(Top, 8));
              Add('  Right : $' + IntToHex(Right, 8));
              Add('  Bottom: $' + IntToHex(Bottom, 8));
            end;
          end
          else
          begin
            Add('  Width  : $' + IntToHex(Width, 8));
            Add('  Height : $' + IntToHex(Height, 8));
            Add('  PalData: $' + IntToHex(PalData, 8));
            Add('  PixData: $' + IntToHex(PixData, 8));
            Add('  Count  : ' + IntToStr(High(TexDesc) - Low(TexDesc) + 1));
          end;
        end;
      end;
      if (PalData > 0) then
      begin
        Pal.Size := SizeOf(TRGBTriple) * 256;
        BdfStream.Position := PalData;
        BdfStream.ReadBuffer(Pal.Memory^, Pal.Size);
        Pix.Size := Width * Height * SizeOf(Byte);
        BdfStream.Position := PixData;
        BdfStream.ReadBuffer(Pix.Memory^, Pix.Size);
        ViewImage.Picture.Bitmap.Handle :=
          Pal888ToBitmap(Pix, Pal, Width, Height);
      end
      else
      begin
        Pix.Size := Width * Height * SizeOf(Word);
        BdfStream.Position := PixData;
        BdfStream.ReadBuffer(Pix.Memory^, Pix.Size);
        ViewImage.Picture.Bitmap.Handle :=
          Rgb565ToBitmap(Pix, Width, Height);
      end;
      if (Node.Parent <> nil) then
      begin
        Pal.Clear;
        Pal.Size := SizeOf(TRGBTriple) * 256;
        FillChar(Pal.Memory^, Pal.Size, 0);
        Rgb := Pal.Memory;
        Inc(Rgb);
        Rgb.rgbtBlue := MAXBYTE;
        Pix.Clear;
        Pix.Size := Width * Height;
        FillChar(Pix.Memory^, Pix.Size, 0);
        Dsc := TexDesc[Node.Index];
        with Dsc do
        begin
          Left := Left * Longword(Width) div $100;
          Top := Top * Longword(Height) div $100;
          Right := Right * Longword(Width) div $100;
          Bottom := Bottom * Longword(Height) div $100;
          FillChar(
            PByte(Cardinal(Pix.Memory) + Top * Longword(Width) + Left)^,
            Right - Left + 1, 1);
          Dat := PByte(Cardinal(Pix.Memory) + Top * Longword(Width) + Left);
          for Idx := Top to Bottom do
          begin
            Dat^ := 1;
            PByte(Cardinal(Dat) + Right - Left)^ := 1;
            Inc(Dat, Width);
          end;
          FillChar(
            PByte(Cardinal(Pix.Memory) + Bottom * Longword(Width) + Left)^,
            Right - Left + 1, 1);
        end;
        RectImage.Picture.Bitmap.Handle :=
          Pal888ToBitmap(Pix, Pal, Width, Height);
        PByte(RectImage.Picture.Bitmap.ScanLine[0])^ := 0;
      end;
    end;
    ImpoButton.Enabled := True;
    ExpoButton.Enabled := True;
  finally
    InfoMemo.Lines.EndUpdate();
    Pix.Free;
    Pal.Free;
  end;
end;

procedure TMainForm.ImpoButtonClick(Sender: TObject);
var
  Node: TTreeNode;
  Idx: Integer;
  Imp: TMemoryStream;
  Pal: TMemoryStream;
  Pix: TMemoryStream;
begin
  Node := TexsTreeView.Selected;
  if (Node = nil) then
    Exit;
  while (Node.Parent <> nil) do
    Node := Node.Parent;
  Idx := Integer(Node.Data);
  if (Idx < Low(Textures)) or (Idx > High(Textures)) then
    Exit;
  if (not DirectoryExists(ImpoDialog.InitialDir)) then
    ImpoDialog.InitialDir := EmptyStr;
  ImpoDialog.FileName := Node.Text;
  with ImpoDialog do
    case FilterIndex of
      2: DefaultExt := 'bmp';
    else
      DefaultExt := 'tga';
    end;
  if (not ImpoDialog.Execute()) then
    Exit;
  Imp := TMemoryStream.Create();
  Pal := TMemoryStream.Create();
  Pix := TMemoryStream.Create();
  try
    Imp.LoadFromFile(ImpoDialog.FileName);
    with Textures[Idx] do
    begin
      if (PalData > 0) then
      begin
        case ImpoDialog.FilterIndex of
          2: BmpPalToPal888(Imp, Pix, Pal, Width, Height);
        else
          TgaPalToPal888(Imp, Pix, Pal, Width, Height);
        end;
        if (Pal.Size <> SizeOf(TRGBTriple) * 256) or
          (Pix.Size <> Width * Height) then
          ShowMessage('Internal error')
        else
        begin
          BdfStream.Position := PalData;
          BdfStream.WriteBuffer(Pal.Memory^, Pal.Size);
          BdfStream.Position := PixData;
          BdfStream.WriteBuffer(Pix.Memory^, Pix.Size);
        end;
      end
      else
      begin
        case ImpoDialog.FilterIndex of
          2: Bmp888ToRgb565(Imp, Pix, Width, Height);
        else
          Tga888ToRgb565(Imp, Pix, Width, Height);
        end;
        if (Pix.Size <> Width * Height * SizeOf(Word)) then
          ShowMessage('Internal error')
        else
        begin
          BdfStream.Position := PixData;
          BdfStream.WriteBuffer(Pix.Memory^, Pix.Size);
        end;
      end;
      TexsTreeViewChange(nil, Node);
    end;
  finally
    Pix.Free;
    Pal.Free;
    Imp.Free;
  end;
end;

procedure TMainForm.ImpoDialogTypeChange(Sender: TObject);
begin
  with ImpoDialog do
    case FilterIndex of
      2: DefaultExt := 'bmp';
    else
      DefaultExt := 'tga';
    end;
end;

procedure TMainForm.ExpoButtonClick(Sender: TObject);
var
  Node: TTreeNode;
  Idx: Integer;
  Pal: TMemoryStream;
  Pix: TMemoryStream;
  Exp: TMemoryStream;
begin
  Node := TexsTreeView.Selected;
  if (Node = nil) then
    Exit;
  while (Node.Parent <> nil) do
    Node := Node.Parent;
  Idx := Integer(Node.Data);
  if (Idx < Low(Textures)) or (Idx > High(Textures)) then
    Exit;
  if (not DirectoryExists(ExpoDialog.InitialDir)) then
    ExpoDialog.InitialDir := EmptyStr;
  ExpoDialog.FileName := Node.Text;
  with ExpoDialog do
    case FilterIndex of
      2: DefaultExt := 'bmp';
    else
      DefaultExt := 'tga';
    end;
  if (not ExpoDialog.Execute()) then
    Exit;
  Pal := TMemoryStream.Create();
  Pix := TMemoryStream.Create();
  Exp := TMemoryStream.Create();
  try
    with Textures[Idx] do
    begin
      if (PalData > 0) then
      begin
        Pal.Size := SizeOf(TRGBTriple) * 256;
        BdfStream.Position := PalData;
        BdfStream.ReadBuffer(Pal.Memory^, Pal.Size);
        Pix.Size := Width * Height * SizeOf(Byte);
        BdfStream.Position := PixData;
        BdfStream.ReadBuffer(Pix.Memory^, Pix.Size);
        case ExpoDialog.FilterIndex of
          2: Pal888ToBmpPal(Pix, Pal, Exp, Width, Height);
        else
          Pal888ToTgaPal(Pix, Pal, Exp, Width, Height);
        end;
      end
      else
      begin
        Pix.Size := Width * Height * SizeOf(Word);
        BdfStream.Position := PixData;
        BdfStream.ReadBuffer(Pix.Memory^, Pix.Size);
        case ExpoDialog.FilterIndex of
          2: Rgb565ToBmp888(Pix, Exp, Width, Height);
        else
          Rgb565ToTga888(Pix, Exp, Width, Height);
        end;
      end;
      Exp.SaveToFile(ExpoDialog.FileName);
    end;
  finally
    Exp.Free;
    Pix.Free;
    Pal.Free;
  end;
end;

procedure TMainForm.ExpoDialogTypeChange(Sender: TObject);
begin
  with ExpoDialog do
    case FilterIndex of
      2: DefaultExt := 'bmp';
    else
      DefaultExt := 'tga';
    end;
end;

end.
