////////////////////////////////////////////////////////////////////////////////
//
//                        Borland Delphi Runtime Library
//                       Planet of Death (POD) Utilities
//
// The original file is: PodUtils.pas, released 2004-09-01.
// The initial developer of the code is: Nico Bendlin <nicode@gmx.net>.
//
// Portions created by Nico Bendlin are
// Copyright (c) 2004 Nico Bendlin.
//
// The contents of this file are used with permission, subject to the Mozilla
// Public License Version 1.1 (the "License"); you may not use this file except
// in compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/MPL-1.1.html
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
// the specific language governing rights and limitations under the License.
//
// Alternatively, the contents of this file may be used under the terms of
// either the GNU General Public License Version 2 or later (the "GPL"), or
// the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
// in which case the provisions of the GPL or the LGPL are applicable instead
// of those above. If you wish to allow use of your version of this file only
// under the terms of either the GPL or the LGPL, and not to allow others to
// use your version of this file under the terms of the MPL, indicate your
// decision by deleting the provisions above and replace them with the notice
// and other provisions required by the GPL or the LGPL. If you do not delete
// the provisions above, a recipient may use your version of this file under
// the terms of any one of the MPL, the GPL or the LGPL.
//
////////////////////////////////////////////////////////////////////////////////
//
//  Notes:  First public version.  Not yet optimized for speed - but works :)
//
////////////////////////////////////////////////////////////////////////////////

unit PodUtils;

{.$HINTS OFF}
{.$WARNINGS OFF}
{$RANGECHECKS OFF}
{$OVERFLOWCHECKS OFF}
{$EXTENDEDSYNTAX ON}

interface

uses
  Windows, Classes;


// RGB 565

procedure Rgb565ToTga888(Source, Dest: TStream; Width, Height: Longint);
procedure Tga888ToRgb565(Source, Dest: TStream; Width, Height: Longint);
procedure Rgb565ToBmp888(Source, Dest: TStream; Width, Height: Longint);
procedure Bmp888ToRgb565(Source, Dest: TStream; Width, Height: Longint);
function Rgb565ToBitmap(Source: TStream; Width, Height: Longint): HBITMAP;

// PAL 888

procedure Pal888ToTgaPal(Pix, Pal, Dest: TStream; Width, Height: Longint);
procedure TgaPalToPal888(Source, Pix, Pal: TStream; Width, Height: Longint);
procedure Pal888ToBmpPal(Pix, Pal, Dest: TStream; Width, Height: Longint);
procedure BmpPalToPal888(Source, Pix, Pal: TStream; Width, Height: Longint);
function Pal888ToBitmap(Pix, Pal: TStream; Width, Height: Longint): HBITMAP;

implementation

uses
  SysUtils, PodConst;


type
  TTgaFileHeader = packed record
    tgaIdentSize: Byte;
    tgaMapType  : Byte;
    tgaType     : Byte;
    tgaMapStart : Word;
    tgaMapLength: Word;
    tgaMapBits  : Byte;
    tgaXStart   : Word;
    tgaYStart   : Word;
    tgaWidth    : Word;
    tgaHeight   : Word;
    tgaBits     : Byte;
    tgaDesc     : Byte;
  end;

const
  MAXBIT5 = $1F;
  MAXBIT6 = $3F;

// RGB 565

procedure Rgb565ToTga888(Source, Dest: TStream; Width, Height: Longint);
var
  TgaHeader: TTgaFileHeader;
  ScanLines: array of Pointer;
  ScanBytes: Integer;
  X: Integer;
  Y: Integer;
  Pixel565: Word;
  Pixel888: PRGBTriple;
begin
  FillChar(TgaHeader, SizeOf(TTgaFileHeader), 0);
  with TgaHeader do
  begin
    tgaType   := 2;
    tgaWidth  := Width;
    tgaHeight := Height;
    tgaBits   := 24;
  end;
  Dest.WriteBuffer(TgaHeader, SizeOf(TTgaFileHeader));
  SetLength(ScanLines, Height);
  for Y := 0 to Height - 1 do
    ScanLines[Y] := nil;
  try
    ScanBytes := Width * SizeOf(TRGBTriple);
    for Y := 0 to Height - 1 do
      GetMem(ScanLines[Y], ScanBytes);
    for Y := 0 to Height - 1 do
    begin
      Pixel888 := ScanLines[Y];
      for X := 0 to Width - 1 do
      begin
        Source.ReadBuffer(Pixel565, SizeOf(Word));
        with Pixel888^ do
        begin
          rgbtRed   := Pixel565 and (MAXBIT5 shl 11) shr (11 - 3);
          rgbtGreen := Pixel565 and (MAXBIT6 shl  5) shr ( 5 - 2);
          rgbtBlue  := Pixel565 and (MAXBIT5       ) shl (     3);
        end;
        Inc(Pixel888);
      end;
    end;
    for Y := Height - 1 downto 0 do
      Dest.WriteBuffer(ScanLines[Y]^, ScanBytes);
  finally
    for Y := 0 to Height - 1 do
      if (ScanLines[Y] <> nil) then
        FreeMem(ScanLines[Y]);
  end;
end;

procedure Tga888ToRgb565(Source, Dest: TStream; Width, Height: Longint);
var
  TgaHeader: TTgaFileHeader;
  ScanLines: array of Pointer;
  ScanBytes: Integer;
  Skip: Byte;
  X: Integer;
  Y: Integer;
  Pixel565: PWord;
  Pixel888: TRGBTriple;
begin
  Source.ReadBuffer(TgaHeader, SizeOf(TTgaFileHeader));
  with TgaHeader do
  begin
    if (tgaMapType <> 0) or (tgaType <> 2) or
      (tgaMapStart <> 0) or (tgaMapLength <> 0) or (tgaMapBits <> 0) or
      (tgaBits <> 24) or (tgaDesc and not $20 <> 0) then
      raise EStreamError.CreateResFmt(
        PResStringRec(@SPodImpImageFrmtError), ['Uncompressed RGB888']);
    if (tgaWidth <> Width) or (tgaHeight <> Height) then
      raise EStreamError.CreateResFmt(
        PResStringRec(@SPodImpImageSizeError), [Width, Height]);
    for X := 1 to tgaIdentSize do
      Source.ReadBuffer(Skip, SizeOf(Byte));
  end;
  SetLength(ScanLines, Height);
  for Y := 0 to Height - 1 do
    ScanLines[Y] := nil;
  try
    ScanBytes := Width * SizeOf(Word);
    for Y := 0 to Height - 1 do
      GetMem(ScanLines[Y], ScanBytes);
    for Y := 0 to Height - 1 do
    begin
      Pixel565 := ScanLines[Y];
      for X := 0 to Width - 1 do
      begin
        Source.ReadBuffer(Pixel888, SizeOf(TRGBTriple));
        with Pixel888 do
          Pixel565^ :=
            (Word(rgbtRed   shr 3) shl 11) or
            (Word(rgbtGreen shr 2) shl  5) or
            (Word(rgbtBlue  shr 3)       );
        Inc(Pixel565);
      end;
    end;
    if (TgaHeader.tgaDesc and $20 = $20) then
      for Y := 0 to Height - 1 do
        Dest.WriteBuffer(ScanLines[Y]^, ScanBytes)
    else
      for Y := Height - 1 downto 0 do
        Dest.WriteBuffer(ScanLines[Y]^, ScanBytes);
  finally
    for Y := 0 to Height - 1 do
      if (ScanLines[Y] <> nil) then
        FreeMem(ScanLines[Y]);
  end;
end;

procedure Rgb565ToBmp888(Source, Dest: TStream; Width, Height: Longint);
var
  FileHeader: TBitmapFileHeader;
  InfoHeader: TBitmapInfoHeader;
  X: Integer;
  Y: Integer;
  ScanBytes: Integer;
  ScanLines: array of Pointer;
  Pixel565: Word;
  Pixel888: PRGBTriple;
begin
  ScanBytes := (Width * SizeOf(TRGBTriple) + 3) and not 3;
  FillChar(FileHeader, SizeOf(TBitmapFileHeader), 0);
  with FileHeader do
  begin
    bfType := $4D42;  // 'BM'
    bfSize := SizeOf(TBitmapFileHeader) + SizeOf(TBitmapInfoHeader) +
      Height * ScanBytes;
    bfOffBits := SizeOf(TBitmapFileHeader) + SizeOf(TBitmapInfoHeader);
  end;
  Dest.WriteBuffer(FileHeader, SizeOf(TBitmapFileHeader));
  FillChar(InfoHeader, SizeOf(TBitmapInfoHeader), 0);
  with InfoHeader do
  begin
    biSize := SizeOf(TBitmapInfoHeader);
    biWidth := Width;
    biHeight := Height;
    biPlanes := 1;
    biBitCount := 24;
    biSizeImage := Height * ScanBytes;
    biCompression := BI_RGB;
  end;
  Dest.WriteBuffer(InfoHeader, SizeOf(TBitmapInfoHeader));
  SetLength(ScanLines, Height);
  for Y := 0 to Height - 1 do
    ScanLines[Y] := nil;
  try
    for Y := 0 to Height - 1 do
      ScanLines[Y] := AllocMem(ScanBytes);
    for Y := 0 to Height - 1 do
    begin
      Pixel888 := ScanLines[Y];
      for X := 0 to Width - 1 do
      begin
        Source.ReadBuffer(Pixel565, SizeOf(Word));
        with Pixel888^ do
        begin
          rgbtRed   := Pixel565 and (MAXBIT5 shl 11) shr (11 - 3);
          rgbtGreen := Pixel565 and (MAXBIT6 shl  5) shr ( 5 - 2);
          rgbtBlue  := Pixel565 and (MAXBIT5       ) shl (     3);
        end;
        Inc(Pixel888);
      end;
    end;
    for Y := Height - 1 downto 0 do
      Dest.WriteBuffer(ScanLines[Y]^, ScanBytes);
  finally
    for Y := 0 to Height - 1 do
      if (ScanLines[Y] <> nil) then
        FreeMem(ScanLines[Y]);
  end;
end;

procedure Bmp888ToRgb565(Source, Dest: TStream; Width, Height: Longint);
var
  FileHeader: TBitmapFileHeader;
  InfoHeader: TBitmapInfoHeader;
  ScanLines: array of Pointer;
  ScanBytes: Integer;
  Flip: Boolean;
  Skip: Byte;
  X: Integer;
  Y: Integer;
  Pixel565: PWord;
  Pixel888: TRGBTriple;
begin
  Source.ReadBuffer(FileHeader, SizeOf(TBitmapFileHeader));
  if (FileHeader.bfType <> $4D42) then  // 'BM'
    raise EStreamError.CreateResFmt(
      PResStringRec(@SPodImpImageFrmtError), ['Windows Bitmap']);
  Source.ReadBuffer(InfoHeader, SizeOf(TBitmapInfoHeader));
  with InfoHeader do
  begin
    if (biBitCount <> 24) or (biCompression <> BI_RGB) then
      raise EStreamError.CreateResFmt(
        PResStringRec(@SPodImpImageFrmtError), ['Uncompressed RGB888']);
    Flip := (biHeight < 0);
    if Flip then
      biHeight := -biHeight;
    if (biWidth <> Width) or (biHeight <> Height) then
      raise EStreamError.CreateResFmt(
        PResStringRec(@SPodImpImageSizeError), [Width, Height]);
  end;
  for X := 1 to FileHeader.bfOffBits -
    SizeOf(TBitmapFileHeader) - SizeOf(TBitmapInfoHeader) do
    Source.ReadBuffer(Skip, SizeOf(Byte));
  SetLength(ScanLines, Height);
  for Y := 0 to Height - 1 do
    ScanLines[Y] := nil;
  try
    ScanBytes := Width * SizeOf(Word);
    for Y := 0 to Height - 1 do
      GetMem(ScanLines[Y], ScanBytes);
    for Y := 0 to Height - 1 do
    begin
      Pixel565 := ScanLines[Y];
      for X := 0 to Width - 1 do
      begin
        Source.ReadBuffer(Pixel888, SizeOf(TRGBTriple));
        with Pixel888 do
          Pixel565^ :=
            (Word(rgbtRed   shr 3) shl 11) or
            (Word(rgbtGreen shr 2) shl  5) or
            (Word(rgbtBlue  shr 3)       );
        Inc(Pixel565);
      end;
      for X := 1 to Width * SizeOf(TRGBTriple) and 3 do
        Source.ReadBuffer(Skip, SizeOf(Byte));
    end;
    if Flip then
      for Y := 0 to Height - 1 do
        Dest.WriteBuffer(ScanLines[Y]^, ScanBytes)
    else
      for Y := Height - 1 downto 0 do
        Dest.WriteBuffer(ScanLines[Y]^, ScanBytes);
  finally
    for Y := 0 to Height - 1 do
      if (ScanLines[Y] <> nil) then
        FreeMem(ScanLines[Y]);
  end;
end;

function Rgb565ToBitmap(Source: TStream; Width, Height: Longint): HBITMAP;
var
  BitmapInfo: TBitmapInfo;
  X: Integer;
  Y: Integer;
  ScanBytes: Integer;
  Pixel565: Word;
  Pixel888: PRGBTriple;
begin
  ScanBytes := (Width * SizeOf(TRGBTriple) + 3) and not 3;
  FillChar(BitmapInfo, SizeOf(TBitmapInfo), 0);
  with BitmapInfo.bmiHeader do
  begin
    biSize := SizeOf(TBitmapInfoHeader);
    biWidth := Width;
    biHeight := -Height;
    biPlanes := 1;
    biBitCount := 24;
    biSizeImage := Height * ScanBytes;
    biCompression := BI_RGB;
  end;
  Result := CreateDIBSection(0, BitmapInfo, DIB_RGB_COLORS, Pointer(Pixel888),
    0, 0);
  if (Result <> 0) and (Pixel888 <> nil) then
    try
      for Y := 0 to Height - 1 do
      begin
        Pixel888 := PRGBTriple((Cardinal(Pixel888) + 3) and not 3);
        for X := 0 to Width - 1 do
        begin
          Source.ReadBuffer(Pixel565, SizeOf(Word));
          with Pixel888^ do
          begin
            rgbtRed   := Pixel565 and (MAXBIT5 shl 11) shr (11 - 3);
            rgbtGreen := Pixel565 and (MAXBIT6 shl  5) shr ( 5 - 2);
            rgbtBlue  := Pixel565 and (MAXBIT5       ) shl (     3);
          end;
          Inc(Pixel888);
        end;
      end;
    except
      DeleteObject(Result);
      Result := 0;
    end;
end;

// PAL 888

type
  PBitmapInfo256 = ^TBitmapInfo256;
  TBitmapInfo256 = packed record
    BitmapInfo: TBitmapInfo;
    ColorMap  : array [0..254] of TRGBQuad;
  end;

procedure Pal888ToTgaPal(Pix, Pal, Dest: TStream; Width, Height: Longint);
var
  TgaHeader: TTgaFileHeader;
  ScanLines: array of Pointer;
  X: Integer;
  Y: Integer;
  Pal888: TRGBTriple;
  Temp: Byte;
begin
  FillChar(TgaHeader, SizeOf(TTgaFileHeader), 0);
  with TgaHeader do
  begin
    tgaMapType   := 1;
    tgaType      := 1;
    tgaMapLength := 256;
    tgaMapBits   := 24;
    tgaWidth     := Width;
    tgaHeight    := Height;
    tgaBits      := 8;
  end;
  Dest.WriteBuffer(TgaHeader, SizeOf(TTgaFileHeader));
  SetLength(ScanLines, Height);
  for Y := 0 to Height - 1 do
    ScanLines[Y] := nil;
  try
    for X := 0 to TgaHeader.tgaMapLength - 1 do
    begin
      Pal.ReadBuffer(Pal888, SizeOf(TRGBTriple));
      Temp := Pal888.rgbtRed;
      Pal888.rgbtRed := Pal888.rgbtBlue;
      Pal888.rgbtBlue := Temp;
      Dest.WriteBuffer(Pal888, SizeOf(TRGBTriple));
    end;
    for Y := 0 to Height - 1 do
      GetMem(ScanLines[Y], Width);
    for Y := 0 to Height - 1 do
      Pix.ReadBuffer(ScanLines[Y]^, Width);
    for Y := Height - 1 downto 0 do
      Dest.WriteBuffer(ScanLines[Y]^, Width);
  finally
    for Y := 0 to Height - 1 do
      if (ScanLines[Y] <> nil) then
        FreeMem(ScanLines[Y]);
  end;
end;

procedure TgaPalToPal888(Source, Pix, Pal: TStream; Width, Height: Longint);
var
  TgaHeader: TTgaFileHeader;
  ScanLines: array of Pointer;
  Skip: Byte;
  X: Integer;
  Y: Integer;
  Pal888: TRGBTriple;
begin
  Source.ReadBuffer(TgaHeader, SizeOf(TTgaFileHeader));
  with TgaHeader do
  begin
    if (tgaMapType <> 1) or (tgaType <> 1) or
      (tgaMapStart > 255) or (tgaMapLength = 0) or
      (tgaMapStart + tgaMapLength > 256) or
      (tgaMapBits <> 24) or (tgaBits <> 8) or (tgaDesc and not $20 <> 0) then
      raise EStreamError.CreateResFmt(
        PResStringRec(@SPodImpImageFrmtError), ['Uncompressed PAL888']);
    if (tgaWidth <> Width) or (tgaHeight <> Height) then
      raise EStreamError.CreateResFmt(
        PResStringRec(@SPodImpImageSizeError), [Width, Height]);
    for X := 1 to tgaIdentSize do
      Source.ReadBuffer(Skip, SizeOf(Byte));
  end;
  FillChar(Pal888, SizeOf(TRGBTriple), 0);
  for X := 1 to TgaHeader.tgaMapStart do
    Pal.WriteBuffer(Pal888, SizeOf(TRGBTriple));
  for X := 1 to TgaHeader.tgaMapLength do
  begin
    Source.ReadBuffer(Pal888, SizeOf(TRGBTriple));
    with Pal888 do
    begin
      Skip := rgbtRed;
      Pal888.rgbtRed := Pal888.rgbtBlue;
      Pal888.rgbtBlue := Skip;
    end;
    Pal.WriteBuffer(Pal888, SizeOf(TRGBTriple));
  end;
  FillChar(Pal888, SizeOf(TRGBTriple), 0);
  for X := TgaHeader.tgaMapStart + TgaHeader.tgaMapLength to 255 do
    Pal.WriteBuffer(Pal888, SizeOf(TRGBTriple));
  SetLength(ScanLines, Height);
  for Y := 0 to Height - 1 do
    ScanLines[Y] := nil;
  try
    for Y := 0 to Height - 1 do
      GetMem(ScanLines[Y], Width);
    for Y := 0 to Height - 1 do
      Source.ReadBuffer(ScanLines[Y]^, Width);
    if (TgaHeader.tgaDesc and $20 = $20) then
      for Y := 0 to Height - 1 do
        Pix.WriteBuffer(ScanLines[Y]^, Width)
    else
      for Y := Height - 1 downto 0 do
        Pix.WriteBuffer(ScanLines[Y]^, Width);
  finally
    for Y := 0 to Height - 1 do
      if (ScanLines[Y] <> nil) then
        FreeMem(ScanLines[Y]);
  end;
end;

procedure Pal888ToBmpPal(Pix, Pal, Dest: TStream; Width, Height: Longint);
var
  FileHeader: TBitmapFileHeader;
  BmpInfo256: TBitmapInfo256;
  X: Integer;
  Y: Integer;
  ScanBytes: Integer;
  ScanLines: array of Pointer;
  Pal888: TRGBTriple;
begin
  ScanBytes := (Width + 3) and not 3;
  FillChar(FileHeader, SizeOf(TBitmapFileHeader), 0);
  with FileHeader do
  begin
    bfType := $4D42;  // 'BM'
    bfSize := SizeOf(TBitmapFileHeader) + SizeOf(TBitmapInfo256) +
      Height * ScanBytes;
    bfOffBits := SizeOf(TBitmapFileHeader) + SizeOf(TBitmapInfo256);
  end;
  Dest.WriteBuffer(FileHeader, SizeOf(TBitmapFileHeader));
  FillChar(BmpInfo256, SizeOf(TBitmapInfo256), 0);
  with BmpInfo256.BitmapInfo.bmiHeader do
  begin
    biSize := SizeOf(TBitmapInfoHeader);
    biWidth := Width;
    biHeight := Height;
    biPlanes := 1;
    biBitCount := 8;
    biCompression := BI_RGB;
    biSizeImage := Height * ScanBytes;
    biClrUsed := 256;
    biClrImportant := 256;
  end;
  for X := 0 to 255 do
    with BmpInfo256.BitmapInfo.bmiColors[X] do
    begin
      Pal.ReadBuffer(Pal888, SizeOf(Pal888));
      rgbBlue := Pal888.rgbtRed;
      rgbGreen := Pal888.rgbtGreen;
      rgbRed := Pal888.rgbtBlue;
    end;
  Dest.WriteBuffer(BmpInfo256, SizeOf(TBitmapInfo256));
  SetLength(ScanLines, Height);
  for Y := 0 to Height - 1 do
    ScanLines[Y] := nil;
  try
    for Y := 0 to Height - 1 do
      ScanLines[Y] := AllocMem(ScanBytes);
    for Y := 0 to Height - 1 do
      Pix.ReadBuffer(ScanLines[Y]^, Width);
    for Y := Height - 1 downto 0 do
      Dest.WriteBuffer(ScanLines[Y]^, ScanBytes);
  finally
    for Y := 0 to Height - 1 do
      if (ScanLines[Y] <> nil) then
        FreeMem(ScanLines[Y]);
  end;
end;

procedure BmpPalToPal888(Source, Pix, Pal: TStream; Width, Height: Longint);
var
  FileHeader: TBitmapFileHeader;
  InfoHeader: TBitmapInfoHeader;
  ScanLines: array of Pointer;
  Flip: Boolean;
  Skip: Byte;
  X: Integer;
  Y: Integer;
  Pal8888: TRGBQuad;
begin
  Source.ReadBuffer(FileHeader, SizeOf(TBitmapFileHeader));
  if (FileHeader.bfType <> $4D42) then  // 'BM'
    raise EStreamError.CreateResFmt(
      PResStringRec(@SPodImpImageFrmtError), ['Windows Bitmap']);
  Source.ReadBuffer(InfoHeader, SizeOf(TBitmapInfoHeader));
  with InfoHeader do
  begin
    if (biBitCount <> 8) or (biCompression <> BI_RGB) or
      (biClrUsed = 0) or (biClrUsed > 256) then
      raise EStreamError.CreateResFmt(
        PResStringRec(@SPodImpImageFrmtError), ['Uncompressed PAL8888']);
    Flip := (biHeight < 0);
    if Flip then
      biHeight := -biHeight;
    if (biWidth <> Width) or (biHeight <> Height) then
      raise EStreamError.CreateResFmt(
        PResStringRec(@SPodImpImageSizeError), [Width, Height]);
  end;
  for X := 1 to InfoHeader.biClrUsed do
  begin
    Source.ReadBuffer(Pal8888, SizeOf(TRGBQuad));
    with Pal8888 do
    begin
      Skip := rgbRed;
      rgbRed := rgbBlue;
      rgbBlue := Skip;
    end;
    Pal.WriteBuffer(Pal8888, SizeOf(TRGBTriple));
  end;
  FillChar(Pal8888, SizeOf(TRGBQuad), 0);
  for X := InfoHeader.biClrUsed to 255 do
    Pal.WriteBuffer(Pal8888, SizeOf(TRGBTriple));
  for X := 1 to FileHeader.bfOffBits -
    SizeOf(TBitmapFileHeader) - SizeOf(TBitmapInfoHeader) -
    InfoHeader.biClrUsed * SizeOf(TRGBQuad) do
    Source.ReadBuffer(Skip, SizeOf(Byte));
  SetLength(ScanLines, Height);
  for Y := 0 to Height - 1 do
    ScanLines[Y] := nil;
  try
    for Y := 0 to Height - 1 do
      GetMem(ScanLines[Y], Width);
    for Y := 0 to Height - 1 do
    begin
      Source.ReadBuffer(ScanLines[Y]^, Width);
      for X := 1 to Width and 3 do
        Source.ReadBuffer(Skip, SizeOf(Byte));
    end;
    if Flip then
      for Y := 0 to Height - 1 do
        Pix.WriteBuffer(ScanLines[Y]^, Width)
    else
      for Y := Height - 1 downto 0 do
        Pix.WriteBuffer(ScanLines[Y]^, Width);
  finally
    for Y := 0 to Height - 1 do
      if (ScanLines[Y] <> nil) then
        FreeMem(ScanLines[Y]);
  end;
end;

function Pal888ToBitmap(Pix, Pal: TStream; Width, Height: Longint): HBITMAP;
var
  BitmapInfo256: TBitmapInfo256;
  ScanBytes: Integer;
  Pal888: TRGBTriple;
  X: Integer;
  Y: Integer;
  Pixel: PByte;
begin
  ScanBytes := (Width + 3) and not 3;
  FillChar(BitmapInfo256, SizeOf(TBitmapInfo), 0);
  with BitmapInfo256.BitmapInfo.bmiHeader do
  begin
    biSize := SizeOf(TBitmapInfoHeader);
    biWidth := Width;
    biHeight := -Height;
    biPlanes := 1;
    biBitCount := 8;
    biCompression := BI_RGB;
    biSizeImage := Height * ScanBytes;
    biClrUsed := 256;
    biClrImportant := 256;
  end;
  for X := 0 to 255 do
    with BitmapInfo256.BitmapInfo.bmiColors[X] do
    begin
      Pal.ReadBuffer(Pal888, SizeOf(Pal888));
      rgbBlue := Pal888.rgbtRed;
      rgbGreen := Pal888.rgbtGreen;
      rgbRed := Pal888.rgbtBlue;
    end;
  Result := CreateDIBSection(0, BitmapInfo256.BitmapInfo, DIB_RGB_COLORS,
    Pointer(Pixel), 0, 0);
  if (Result <> 0) and (Pixel <> nil) then
    try
      for Y := 0 to Height - 1 do
      begin
        Pixel := PByte((Cardinal(Pixel) + 3) and not 3);
        Pix.ReadBuffer(Pixel^, Width);
        Inc(Pixel, Width);
      end;
    except
      DeleteObject(Result);
      Result := 0;
    end;
end;

end.
